#!/usr/bin/perl

use strict;
use Fcntl qw|O_RDONLY|;

use vars qw|$dumpDir
			$ptrArgs
			$workDir
			$storage|;

use AgentConfig;

use XmlNode;

#
# Constants
#
my $kilobyte = 1024;
my $megabyte = $kilobyte*$kilobyte;
my $kilobittobyte = 1024/8;
my $raqdomains = 'raqdomains';

my $vsitesDir = "/home/sites";
my $spoolDir = "/var/spool/mail";

#
# parse command line
#
my %arg_opts = ('--help|-h'=>'',
		'--get-status|-s'=>'',
		'--dump-accounts|-dc'=>'s',
		'--dump-domains|-dd'=>'s',
		'--dump-all|-da'=>'',
		'--get-content-list|-lc'=>'',
		'--no-content|-nc'=>'',
		'--no-compress|-nz'=>'',
		'--output|-o'=>'s',
		'--status-file|-sf'=>'s',
	       );

my %shadow;

#@@INCLUDE FILE="agent.include.pl"@@
if(-f 'agent.include.pl'){
  require 'agent.include.pl';
}else{
  require '../agent.include.pl';
}
#@@/INCLUDE@@

$ptrArgs = getArguments(\@ARGV, \%arg_opts);

my ($outPath,$dumpFile,$statusFile);

$workDir = AgentConfig::cwd();

$dumpFile = $workDir . '/dump.xml';
$statusFile = $workDir . '/dumping-status.xml';
$dumpDir = $workDir . '/pma';

my $objDumpStatus = &makeDumpStatus($ptrArgs->{'status-file'}||$statusFile);

#
# get MIME Base64 encoder
#
my $wrapBase64 = makeMIMEBase64();

my $version = getVersion();
if ($version) {
  eval "require Cobalt::Fpx";
  eval "require Cobalt::Vsite";
  eval "require Cobalt::Meta";
  eval "require Cobalt::List";
}

if (exists $ptrArgs->{'get-status'}) {
  printAgentStatus();
  exit 0;
}

if(exists ($ptrArgs->{'dump-all'}) ||
   exists ($ptrArgs->{'dump-accounts'}) ||
   exists ($ptrArgs->{'dump-domains'})) {

  my ($root, @accounts, @domains, $ptrAccounts, $ptrDomains, $value);
  initStorage();
  getDumpDir($dumpDir);

  if (exists $ptrArgs->{'no-compress'}) {
	setCompress(0);
  }

  printToLog("Work dir: $workDir");
  printToLog("Dump file: $dumpFile");
  printToLog("Status file: " . $objDumpStatus->{'FILE'}->());

  if ($value = $ptrArgs->{'dump-accounts'}) {
  	if ($value eq "-") {
	  $value = <STDIN>;
	  chomp $value;
	}
    @accounts = split(/\s*,\s*/, $value);
    $ptrAccounts = \@accounts;
  }

  if ($value = $ptrArgs->{'dump-domains'}) {
  	if ($value eq "-") {
	  $value = <STDIN>;
	  chomp $value;
	}
    @domains = split(/\s*,\s*/,$value);
    $ptrDomains = \@domains;
  }
#
# generate a xml dump
#
  $root = getRaQ3Dump($ptrAccounts, $ptrDomains);
#
# print dump to output
#
  $storage->finish($root);
} elsif (exists $ptrArgs->{'get-content-list'}) {
  makeContentList();
} else {
  printHelp();
}

exit 0;

#
# end main
#
#==============================================================
#
# subroutines
#

#
# Domains/Clients dealing
#

my %clientToVsite;
my %vsiteToClient;

sub addVsiteToClient {
  my ($client, $vsite) = @_;


  $clientToVsite{$client} = {} unless exists $clientToVsite{$client};

  $clientToVsite{$client}->{$vsite} = 1;
  $vsiteToClient{$vsite} = $client;

  return 0;
}

sub getCounterpart {
  my ($vsite) = @_;

  if ($vsite =~ /^www\.(.*)$/) {
	return $1;
  } else {
	return "www.$vsite";
  }
}

sub generateObjectsStructure {
  my ($generateClients) = @_;
  my ($domain, $user);

  my @domains = Cobalt::Meta::query("type"=>"vsite", "keys" => ["name", "fqdn"],
									"where" => ["name", "<>", "default"]);

  foreach $domain (@domains) {
	my @users = Cobalt::Meta::query("type"=>"users", "keys"=>["name", "admin"],
									"where"=>["vsite", "=", $domain->get("name")], "sort"=>"name");


	# if we dump only list of domains (--nc), then we should preserve
	# duplicates (domain and www.domain), else suppress them

	unless (exists $ptrArgs->{'no-content'}) {
	  my $counterpart = getCounterpart($domain->get("fqdn"));

	  if (defined $vsiteToClient{$counterpart}) {
		#remove counterpart nahren
		my $c = $vsiteToClient{$counterpart};
		delete $vsiteToClient{$counterpart};
		delete $clientToVsite{$c}->{$counterpart};
		if (scalar keys %{$clientToVsite{$c}} == 0) {
		  delete $clientToVsite{$c};
		}
		next; # next domain in foreach
	  }
	}

	if ($generateClients) {
	  foreach $user (@users) {
		if ($user->get("admin") eq "on") {
		  addVsiteToClient($user->get("name"), $domain->get("fqdn"));
		  last;
		}
	  }
	}

	unless (exists $vsiteToClient{$domain->get("fqdn")}) {
	  addVsiteToClient($raqdomains, $domain->get("fqdn"));
	}
  }
}

sub parseShadow {
  open SHADOW, "/etc/shadow";
  while (<SHADOW>) {
	my @userInfo = split /:/;
	if ($userInfo[1] ne '*') {
	  $shadow{$userInfo[0]} = $userInfo[1];
	}
  }
  close SHADOW;
}

sub getRaQ3Dump {
  my ($ptrAccounts, $ptrDomains) = @_;
  my ($account);

  parseShadow();

  my $root = XmlNode->new('RaQ3dump', 'attributes'=>{'agent-name'=>'RaQ3'});

  generateObjectsStructure(ref($ptrDomains) !~ /ARRAY/);

  if (ref($ptrDomains) =~ /ARRAY/) {
	# suppress duplicates
	my @vsitesToMigrate;
	for my $vsite (@{$ptrDomains}) {
	  if (exists($vsiteToClient{$vsite}) or exists($vsiteToClient{getCounterpart($vsite)})) {
		push @vsitesToMigrate, $vsite;
	  }
	}
    $objDumpStatus->{'COUNTS'}->(0, scalar(@vsitesToMigrate));
    $objDumpStatus->{'PRINT'}->();

	$root->{'ADDCHILD'}->(dumpClient($raqdomains, \@vsitesToMigrate, 0));
  } elsif (ref($ptrAccounts) =~ /ARRAY/) {
    my $clientsCount = scalar(@{$ptrAccounts});
    my $domainsCount = 0;
    foreach $account (@{$ptrAccounts}) {
      $domainsCount += scalar(keys %{$clientToVsite{$account}});
    }
    $objDumpStatus->{'COUNTS'}->($clientsCount, $domainsCount);
    $objDumpStatus->{'PRINT'}->();

	foreach $account (@{$ptrAccounts}) {
	  my @d = keys %{$clientToVsite{$account}};
	  $root->{'ADDCHILD'}->(dumpClient($account, \@d, 1));
	}
  } else {
    my $clientsCount = scalar(keys %clientToVsite);
    my $domainsCount = 0;
    foreach $account (keys %clientToVsite) {
      $domainsCount += scalar(keys %{$clientToVsite{$account}});
    }
    $objDumpStatus->{'COUNTS'}->($clientsCount, $domainsCount);
    $objDumpStatus->{'PRINT'}->();

	foreach $account (keys %clientToVsite) {
	  my @d = keys %{$clientToVsite{$account}};
	  $root->{'ADDCHILD'}->(dumpClient($account, \@d, 1));
	}
  }

  return $root;
}

sub dumpClient {
  my ($clientName, $domains, $countClientInStatus) = @_;
  my ($vsite, $clientNode);

  if ($countClientInStatus) {
    $objDumpStatus->{'ACCOUNT'}->($clientName);
    $objDumpStatus->{'PRINT'}->();
  }

  $clientNode = XmlNode->new('client');

  if ($clientName eq $raqdomains) {
	$clientNode->{'ATTRIBUTE'}->('name', 'raqdomains');
	$clientNode->{'ATTRIBUTE'}->('contact', 'RaQ domains');
	createPasswordNode($clientNode, '', 'plain');
  } else {
	$clientNode->{'ATTRIBUTE'}->('name', $clientName);
	createPasswordNode($clientNode, $shadow{$clientName}, 'encrypted');
  }

  foreach $vsite (@{$domains}) {
	printToLog($vsite);
	my $domain = dumpDomain($vsite);
	if ($domain) {
	  $clientNode->{'ADDCHILD'}->($domain);
	}
  }

  return $clientNode;
}

sub getDomainInfo {
  my ($domainName) = @_;

  my @domainInfo = Cobalt::Meta::query("type"=>"vsite",
									   "keys" => ["name", "ipaddr", "fqdn", "suspend"],
									   "where" => ["fqdn", "=", $domainName]);

  return $domainInfo[0];
}

sub dumpDomain {
  my ($domainName) = @_;

  printToLog("Started dump of $domainName");
  $objDumpStatus->{'DOMAIN'}->($domainName);
  $objDumpStatus->{'PRINT'}->();

  my $domainInfo = getDomainInfo($domainName);
  if (!$domainInfo) {
	$domainInfo = getDomainInfo("www.$domainName");
  }
  if (!$domainInfo) {
	printToLog("Unable to find domain $domainName");
	return;
  }

  my $www = 'false';
  my $name = $domainInfo->get("fqdn");

  my $domainNode = XmlNode->new('domain');

  $domainNode->{'ATTRIBUTE'}->('ip', $domainInfo->get("ipaddr"));
  $domainNode->{'ATTRIBUTE'}->('name', $name);
  $domainNode->{'ATTRIBUTE'}->('www', $www);
  if ($domainInfo->get("suspend") eq "on") {
	$domainNode->{'ATTRIBUTE'}->('enabled', 'false');
  }

  # NO MORE INFO REQUIRED IN '--no-content' MODE
  if (exists $ptrArgs->{'no-content'}) {
	return $domainNode;
  }

  my $domainHandle = $domainInfo->get("name");

  if (-d "$vsitesDir/$domainHandle/web") {
	my $dump = makeDumpFile("$dumpDir/$domainHandle.web", "$vsitesDir/$domainHandle/web");
	if ($dump) {
	  $domainNode->{'ATTRIBUTE'}->('cid', $dump);
	}
  }

  dumpDomainLimits($domainNode, $domainHandle);
  dumpDomainPermissions($domainNode, $domainHandle);
  dumpAnonFtp($domainNode, $domainHandle);
  dumpUsers($domainNode, $name, $domainHandle);
  dumpMailLists($domainNode, $name, $domainHandle);
  dumpCertificate($domainNode, $name, $domainHandle);

  printToLog("<-- finished");

  return $domainNode;
}

sub dumpDomainLimits {
  my ($parent, $domainHandle) = @_;

  my $vsite = Cobalt::Meta->create("type" => "vsite",
								   "keys" => ["quota", "maxusers", "bwlimit", "ipaddr"]);
  $vsite->retrieve($domainHandle);

  if ($vsite->get("quota") != 0) {
	my $limitNode = XmlNode->new('domain-limit', "content" => $vsite->get("quota") * $megabyte);
	$limitNode->{'ATTRIBUTE'}->('name', 'quota');
	$parent->{'ADDCHILD'}->($limitNode);
  }

  if ($vsite->get("maxusers") != 0) {
	my $limitNode = XmlNode->new('domain-limit', "content" => $vsite->get("maxusers"));
	$limitNode->{'ATTRIBUTE'}->('name', 'users');
	$parent->{'ADDCHILD'}->($limitNode);
  }

  if ($vsite->get("bwlimit") eq "on") {
	my $bwobj = Cobalt::Meta->create("type" => "bw");
	$bwobj->retrieve($vsite->get("ipaddr"));
	my $bwlimit = $bwobj->get("bwlimit");

	my $limitNode = XmlNode->new('domain-limit', $bwlimit * $kilobittobyte);
	$limitNode->{'ATTRIBUTE'}->('name', 'traffic');
	$parent->{'ADDCHILD'}->($limitNode);
  }
}

sub dumpDomainPermissions {
  my ($parent, $domainHandle) = @_;

  my $vsite = Cobalt::Meta->create("type" => "vsite",
								   "keys" => ["emaildomain", "shell", "cgi",
											  "ssi", "fpx", "ssl"]);
  $vsite->retrieve($domainHandle);

  if ($version eq 'RaQ4') {
	createPermNode($parent, 'domain', 'email', 'true')
  } else {
	createPermNode($parent, 'domain', 'email', $vsite->get('emaildomain') eq 'on');
  }

  createPermNode($parent, 'domain', 'shell', $vsite->get('shell') eq 'on');
  createPermNode($parent, 'domain', 'cgi', $vsite->get('cgi') eq 'on');
  createPermNode($parent, 'domain', 'ssi', $vsite->get('ssi') eq 'on');
  createPermNode($parent, 'domain', 'frontpage', $vsite->get('fpx') eq 'on');
  createPermNode($parent, 'domain', 'ssl', $vsite->get('ssl') eq 'on');
}

sub dumpAnonFtp {
  my ($parent, $domainHandle) = @_;

  my $anonftp = Cobalt::Meta->create("type" => "vsite", "keys" => ["ftp", "ftpusers", "ftpquota"]);
  $anonftp->retrieve($domainHandle);

  if ($anonftp->get("ftp") eq "on") {
	my $anonFtpNode = XmlNode->new("anonftp");

	my $usersLimitNode = XmlNode->new("anonftp-limit", "content" => $anonftp->get("ftpusers"));
	$usersLimitNode->{'ATTRIBUTE'}->('name', 'users');
	$anonFtpNode->{'ADDCHILD'}->($usersLimitNode);

	my $incomingQuotaNode = XmlNode->new("anonftp-limit", "content" => $anonftp->get("ftpquota")*$kilobyte);
	$incomingQuotaNode->{'ATTRIBUTE'}->('name', 'incoming-quota');
	$anonFtpNode->{'ADDCHILD'}->($incomingQuotaNode);

	if (-d "$vsitesDir/$domainHandle/ftp/incoming") {
	  my $dump = makeDumpFile("$dumpDir/$domainHandle.ftp.incoming", "$vsitesDir/$domainHandle/ftp/incoming");
	  if ($dump) {
		$anonFtpNode->{'ATTRIBUTE'}->('cid', $dump);
	  }
	}

	if (-d "$vsitesDir/$domainHandle/ftp") {
	  my $dump = makeDumpFile("$dumpDir/$domainHandle.ftp", "$vsitesDir/$domainHandle/ftp", ".", "incoming");
	  if ($dump) {
		$anonFtpNode->{'ATTRIBUTE'}->('cid_incoming', $dump);
	  }
	}
	$parent->{'ADDCHILD'}->($anonFtpNode);
  }
}

sub dumpUsers {
  my ($parent, $domainName, $domainHandle) = @_;

  my $domainHaveAdmin = 0;
  my $user;
  foreach $user (Cobalt::Meta::query("type" => "users", "keys" => ["name", "admin"],
									 "where" => ["vsite", "=", "$domainHandle"],
									 "sort" => "name")) {
	my $admin = $user->get("admin") eq "on";
	dumpUser($parent, $domainHandle, $domainName, $user->get("name"), !$domainHaveAdmin && $admin);
	if ($admin) {
	  $domainHaveAdmin = 1;
	}
  }

  unless ($domainHaveAdmin) {
	dumpSyntheticAdmin($parent, $domainHandle, $domainName);
  }
}

sub dumpUser {
  my ($parent, $domainHandle, $domainName, $userName, $admin) = @_;

  my $user = Cobalt::Meta->create("type" => "users",
								  "keys" => ["name", "fullname", "suspend", "fpx", "shell", "quota"]);
  $user->retrieve($userName);

  my $userNode = XmlNode->new($admin ? 'admin' : 'user');

  if ($user->get("fullname")) {
	$userNode->{'ATTRIBUTE'}->('fullname', $user->get("fullname"));
  }

  if ($user->get("suspend") eq "on") {
	$userNode->{'ATTRIBUTE'}->('enabled', 'false');
  }

  my $sysUserNode = makeSysUserNode($user->get("name"), $shadow{$user->get("name")}, "encrypted");
  $sysUserNode->{'ATTRIBUTE'}->('quota', $user->get("quota") * $megabyte);

  $userNode->{'ADDCHILD'}->($sysUserNode);

  createPermNode($userNode, "user", "shell", $user->get("shell") eq "on");
  createPermNode($userNode, "user", "frontpage", $user->get("fpx") eq "on");

  if (-d "$vsitesDir/$domainHandle/users/".$userName."/web") {
	my $dump = makeDumpFile("$dumpDir/".$userName."@"."$domainHandle.web",
							"$vsitesDir/$domainHandle/users/".$userName."/web");
	if ($dump) {
	  $userNode->{'ATTRIBUTE'}->('cid', $dump);
	}
  }

  if (-d "$vsitesDir/$domainHandle/users/".$userName) {
	my $dump = makeDumpFile("$dumpDir/".$userName."@"."$domainHandle.private",
							"$vsitesDir/$domainHandle/users/".$userName,
							".", "web");
	if ($dump) {
	  $userNode->{'ATTRIBUTE'}->('cid_private', $dump);
	}
  }

  dumpMail($userNode, $domainName, $domainHandle, $userName);

  $parent->{'ADDCHILD'}->($userNode);
}

sub dumpSyntheticAdmin {
  my ($parent, $domainHandle, $domainName) = @_;

  my $name = createUserNameFromDomain($domainName);

  my $adminNode = XmlNode->new('admin');
  $adminNode->{'ATTRIBUTE'}->('fullname', $domainName . " admin");
  $adminNode->{'ADDCHILD'}->(makeSysUserNode($name, '', 'plain'));

  my $adminMailNode = XmlNode->new('mail');
  $adminMailNode->{'ATTRIBUTE'}->('mailname', $name);

  $adminNode->{'ADDCHILD'}->($adminMailNode);

  $parent->{'ADDCHILD'}->($adminNode);
}

# Pieces of old-style interface inside

sub dumpMail {
  my ($parent, $domainName, $domainHandle, $userName) = @_;

  my $mailNode = XmlNode->new('mail');
  $mailNode->{'ATTRIBUTE'}->('mailname', $userName);

  my @forwardList = Cobalt::List::alias_get_vacationless($userName);
  if ($forwardList[0] =~ /^[123]\S*\s/) {
	@forwardList = ();
  }

  my $user = Cobalt::Meta->create("type" => "users", "keys" => ["aliases", "vacation"]);
  $user->retrieve($userName);
  my $aliasesStr = $user->get("aliases");
  my @aliases = split /\s+/, $aliasesStr;

  my ($alias, %uniqueAliases);
  foreach $alias (@aliases) {
	$alias =~ s/^\./_/gi;
	$alias =~ s/\.$/_/gi;
	$uniqueAliases{$alias} = 1;
  }

  # RaQ allows aliases identical to mailnames
  delete $uniqueAliases{$userName};

  my $forward;
  foreach $forward (@forwardList) {
	# RaQ allows forwards to itself
	next if ($forward eq $userName . '@' . $domainName);
	# RaQ allows aliases equivalent to the forwards
	if (index($forward, $domainName) != -1) {
	  $forward =~ /^([^@]*)@/;
	  if (exists $uniqueAliases{$1}) {
		next;
	  }
	}

	if ($forward !~ /@/) {
	  $forward .= '@' . $domainName;
	}

	$mailNode->{'ADDCHILD'}->(XmlNode->new('forward', "content" => $forward));
  }

  foreach $alias (keys %uniqueAliases) {
	$mailNode->{'ADDCHILD'}->(XmlNode->new('alias', "content" => $alias));
  }

  my $vacationEnabled = $user->get("vacation") eq "on";
  my $vacationMsg = $wrapBase64->{'ENCODE'}->(Cobalt::Vacation::vacation_get_message($userName));

  if ($vacationEnabled || $vacationMsg) {
	my $autoresponderNode = XmlNode->new("autoresponder", "content" => $vacationMsg);
	$autoresponderNode->{'ATTRIBUTE'}->('enabled', $vacationEnabled ? "true" : "false");
	$mailNode->{'ADDCHILD'}->($autoresponderNode);
  }

  my @extraParams = ('-h');
  if (-f "$spoolDir/$userName") {
	my $dump = makeDumpFile("$dumpDir/$userName"."@"."$domainHandle.inbox", $spoolDir, $userName, undef, \@extraParams);
	if ($dump) {
	  $mailNode->{'ATTRIBUTE'}->('cid_inbox', $dump);
	}
  }
  my $mailboxdir = "$vsitesDir/$domainHandle/users/$userName/mail";
  if (-d $mailboxdir) {
	my $dump = makeDumpFile("$dumpDir/$userName"."@"."$domainHandle.mail", $mailboxdir, undef, undef, \@extraParams);
	if ($dump) {
	  $mailNode->{'ATTRIBUTE'}->('cid', $dump);
	}
  }

  $parent->{'ADDCHILD'}->($mailNode);
}

# All of the sudden old-style interface found on the RaQ3 :)

sub dumpMailLists {
  my ($parent, $domainName, $domainHandle) = @_;

  my $fqdn = (Cobalt::Vsite::vsite_get_bygroup($domainHandle))[1];
  my @mailLists = Cobalt::List::list_lists($domainHandle, $fqdn);
  if ($mailLists[0] =~ /^[134]\S*\s/) {
    @mailLists = ();
  }

  my $list;
  foreach $list (@mailLists) {
    dumpMailList($parent, $domainHandle, $domainName, $fqdn, $list);
  }
}

sub dumpMailList {
  my ($parent, $domainHandle, $domainName, $fqdn, $list) = @_;

  my $mailListNode = XmlNode->new('maillist');
  $mailListNode->{'ATTRIBUTE'}->('name', $list . '_' . $domainHandle);

  my @members = Cobalt::List::list_get_mem($domainHandle, $fqdn, $list);
  if ($members[0] =~ /^[134]\S*\s/) {
    @members = ();
  }

  my $member;
  foreach $member (@members) {
	if ($member =~ /@/) {
      $mailListNode->{'ADDCHILD'}->(XmlNode->new('recipient', "content" => $member));
    } else {
      $mailListNode->{'ADDCHILD'}->(XmlNode->new('recipient', "content" => $member . '@' . $domainName));
    }
  }
  $parent->{'ADDCHILD'}->($mailListNode);
}

sub dumpCertificate {
  my ($parent, $domainHandle, $domainName) = @_;

  my $certsDir = $vsitesDir . "/" . $domainHandle . "/certs";
  my $certificate;
  if ($certificate = readFile($certsDir . "/certificate")) {

	my $certNode = XmlNode->new('certificate');
	$certNode->{'ADDCHILD'}->(XmlNode->new('certificate-data', "content" => $certificate));

	my $privkey = readFile($certsDir . "/key");
	if ($privkey) {
	  $certNode->{'ADDCHILD'}->(XmlNode->new('private-key', "content" => $privkey));
	}
	$parent->{'ADDCHILD'}->($certNode);
  }
}

sub createPermNode {
  my ($parent, $prefix, $name, $value) = @_;

  if ($value) {
	my $permNode = XmlNode->new($prefix . '-permission');
	$permNode->{'ATTRIBUTE'}->('name', $name);
	$parent->{'ADDCHILD'}->($permNode);
  }
}

sub printAgentStatus {
  my $root = XmlNode->new('agent-status');
  unless ($version) {
    my $item = XmlNode->new('wrong-platform', "content" => defined AgentConfig::iconvBin() ? '' : 'no iconv found on the source host');
    $root->{'ADDCHILD'}->($item);
  }
  $root->serialize(\*STDOUT);
}

sub getVersion {
  my %bldHash = ('2700'=>'RaQ1',
				 '2799'=>'RaQ2',
				 '2800'=>'RaQ2',
				 '3000'=>'RaQ3',
				 '3001'=>'RaQ4',
				 '3100'=>'RaQ4',
				 '3500'=>'RaQXTR',
				 '3599'=>'RaQXTR',
				);

  my ($build, $version);

  if ((-T '/etc/build') && open (CONF, "</etc/build")) {
    while (<CONF>) {
      if (/(\d{4,4})/) {
		my $buildCode = $1;
		if (exists($bldHash{$1})) {
		  $build = $1;
		  $version = $bldHash{$1};
		  last;
		}
      }
    }
    close (CONF);
  }

  if ($version =~ /RaQ3/) {
	return $version;
  }
}

sub printHelp {

  print <<"HELP";

Usage:
  $0 <options>

Options:
  -s |--get-status           get status of the agent
  -dc|--dump-accounts=<list> a coma separated list of resellers to dump
  -dd|--dump-domains=<list>  a coma separated list of customers to dump
  -da|--dump-all             make a full dump

  -lc|--get-content-list     get list of content files
  -nc|--no-content           do not make content files
  -nz|--no-compress          do not compress content files

  -h |--help                 this help

HELP
}
